<?php
/* Copyright (C) 2026 Florian Dufourg <florian.dufourg@simple-soft.eu>
 * Copyright (C) 2024		MDW							<mdeweerd@users.noreply.github.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

/**
 * \file    smartbankimport/class/actions_smartbankimport.class.php
 * \ingroup smartbankimport
 * \brief   Example hook overload.
 *
 * Put detailed description here.
 */

require_once DOL_DOCUMENT_ROOT.'/core/class/commonhookactions.class.php';
dol_include_once('/smartbankimport/class/fileslib.class.php');

/**
 * Class ActionsSmartBankImport
 */
class ActionsSmartBankImport extends CommonHookActions
{
	/**
	 * @var DoliDB Database handler.
	 */
	public $db;

	/**
	 * @var string Error code (or message)
	 */
	public $error = '';

	/**
	 * @var array Errors
	 */
	public $errors = array();


	/**
	 * @var array Hook results. Propagated to $hookmanager->resArray for later reuse
	 */
	public $results = array();

	/**
	 * @var ?string String displayed by executeHook() immediately after return
	 */
	public $resprints;

	/**
	 * @var int		Priority of hook (50 is used if value is not defined)
	 */
	public $priority;


	/**
	 * Constructor
	 *
	 *  @param		DoliDB		$db      Database handler
	 */
	public function __construct($db)
	{
		$this->db = $db;
	}


	/**
	 * Overloading the addMoreActionsEmailCollector function : replacing the parent's function with the one below
	 *
	 * @param   array           $parameters     Hook metadatas (context, etc...)
	 * @param   CommonObject    $object         The object to process (an invoice if you are in invoice module, a propale in propale's module, etc...)
	 * @param   string          $action         Current action (if set). Generally create or edit or null
	 * @param   HookManager     $hookmanager    Hook manager propagated to allow calling another hook
	 * @return  int                             Return integer < 0 on error, 0 on success, 1 to replace standard code
	 */
	public function addMoreActionsEmailCollector($parameters, &$object, &$action, $hookmanager)
	{
		global $conf, $user, $langs;

		$this->results = array(
			'hook_sbi_file_lib' => "CollectFilesToSmartBankImportFromEmail"
		);

		return 0;
	}

	/**
	 * Overloading the doCollectImapOneCollector function : replacing the parent's function with the one below
	 *
	 * @param   array           $parameters     Hook metadatas (context, etc...)
	 * @param   CommonObject    $object         The object to process (an invoice if you are in invoice module, a propale in propale's module, etc...)
	 * @param   string          $operationtype         Current operation type
	 * @param   HookManager     $hookmanager    Hook manager propagated to allow calling another hook
	 * @return  int                             Return integer < 0 on error, 0 on success, 1 to replace standard code
	 */
	public function doCollectImapOneCollector($parameters, &$object, &$operationtype, $hookmanager)
	{
		global $conf, $langs;

		//print '<pre>'; print_r("ici"); print '</pre>';exit;


		$conf->setValues($this->db);

		$actionparam_json = str_replace("'", '"', $parameters["actionparam"]);
		$actionparam_json = str_replace(" ", "", $actionparam_json);


		$actionparam = json_decode($actionparam_json);

		$error = 0; // Error counter

		if($operationtype == "hook_sbi_file_lib"){


			//CHECK IF JSON PARAM IS VALID
			
			if(empty($actionparam)){
				$this->resprints = 'invalid json format';
				return -1;
			}

			//CHECK IF USER ID IS DEFINED
			if(empty($actionparam->userid)){
				$this->resprints = 'userid is required';
				return -1;
			}
			
			$user = new User($this->db);
			$user->fetch($actionparam->userid);

			$entity = $user->entity;

			if (empty($entity)) {
				$entity = 1;
			}


			//USE EMAILCOLLECTORCLASS TO SAVE ATTACHMENT
			if ($parameters["attachments"]) {



					$upload_dir = $conf->smartbankimport->multidir_output[$entity].'/lib';

					if (!dol_is_dir($upload_dir)) {
						dol_mkdir($upload_dir);
					}


					foreach ($parameters["attachments"] as $attachment) {

						$filename = dol_sanitizeFileName($attachment->getName());

						$extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

						//CHECK IF USER ID IS DEFINED
						if(!empty($actionparam->fileextension)){
							$acceptedExt = explode(",", $actionparam->fileextension);

							if(!in_array($extension, $acceptedExt)){
								//file not accepted
								continue;
							}
						}

						if(substr($filename, -1) == "="){
							$extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

							$filename = "fromemail_".date("Ymd_His").".".$extension;
						}

						$typeattachment = (string) $attachment->getDisposition();

						$destfile = $upload_dir . "/" . $filename;

						$destfiletmp = DOL_DATA_ROOT.'/admin/temp/'.$filename;

						$res = dol_delete_file($destfiletmp);
						

						if (!dol_is_dir(dirname($upload_dir))) {
							$this->resprints = 'Directory does not exists : '.dirname($upload_dir);
							return -1;
						}

						if (dol_is_file($destfile)) {
							$this->resprints = "File with name '".$destfile."' already exists.";
							return -1;
						}

						// in case temporary directory admin/temp doesn't exist
						if (!dol_is_dir(dirname($destfiletmp))) {
							dol_mkdir(dirname($destfiletmp));
						}

						$newfilecontent = $attachment->getContent();

						$fhandle = @fopen($destfiletmp, 'w');
						if ($fhandle) {
							$nbofbyteswrote = fwrite($fhandle, $newfilecontent);
							fclose($fhandle);
							dolChmod($destfiletmp);
						} else {
							$this->resprints = "Failed to open file '".$destfiletmp."' for write";
							return -1;
						}

						$disablevirusscan = 0;
						$src_file = $destfiletmp;
						$dest_file = $destfile;

						// Security:
						// If we need to make a virus scan
						if (empty($disablevirusscan) && file_exists($src_file)) {
							$checkvirusarray = dolCheckVirus($src_file, $dest_file);
							if (count($checkvirusarray)) {
								dol_syslog('Files.lib::dol_move_uploaded_file File "'.$src_file.'" (target name "'.$dest_file.'") KO with antivirus: errors='.implode(',', $checkvirusarray), LOG_WARNING);
								$this->resprints = 'ErrorFileIsInfectedWithAVirus: '.implode(',', $checkvirusarray);
								return -1;
							}
						}

						// Security:
						// Disallow file with some extensions. We rename them.
						// Because if we put the documents directory into a directory inside web root (very bad), this allows to execute on demand arbitrary code.
						if (isAFileWithExecutableContent($dest_file) && !getDolGlobalString('MAIN_DOCUMENT_IS_OUTSIDE_WEBROOT_SO_NOEXE_NOT_REQUIRED')) {
							// $upload_dir ends with a slash, so be must be sure the medias dir to compare to ends with slash too.
							$publicmediasdirwithslash = $conf->medias->multidir_output[$conf->entity];
							if (!preg_match('/\/$/', $publicmediasdirwithslash)) {
								$publicmediasdirwithslash .= '/';
							}

							if (strpos($upload_dir, $publicmediasdirwithslash) !== 0 || !getDolGlobalInt("MAIN_DOCUMENT_DISABLE_NOEXE_IN_MEDIAS_DIR")) {	// We never add .noexe on files into media directory
								$dest_file .= '.noexe';
							}
						}

						// Security:
						// We refuse cache files/dirs, upload using .. and pipes into filenames.
						if (preg_match('/^\./', basename($src_file)) || preg_match('/\.\./', $src_file) || preg_match('/[<>|]/', $src_file)) {
							dol_syslog("Refused to deliver file ".$src_file, LOG_WARNING);
							$this->resprints = "Refused to deliver file ".$src_file;
							return -1;
						}

						// Security:
						// We refuse cache files/dirs, upload using .. and pipes into filenames.
						if (preg_match('/^\./', basename($dest_file)) || preg_match('/\.\./', $dest_file) || preg_match('/[<>|]/', $dest_file)) {
							dol_syslog("Refused to deliver file ".$dest_file, LOG_WARNING);
							$this->resprints = "Refused to deliver file ".$dest_file;
							return -1;
						}

						$moreinfo = array('note_private' => 'File uploaded using email collector');

						// Move the temporary file at its final emplacement
						$overwriteifexists = 0;
						$result = dol_move($destfiletmp, $dest_file, 0, $overwriteifexists, 1, 1, $moreinfo);
						if (!$result) {
							$this->resprints = "Failed to move file into '".$dest_file."'";
							return -1;
						}

						//WE save file upladed to DB if loaded in lib folder

						$fileLibObject = new FilesLib($this->db);
						$fileLibObject->label = dol_basename($destfile);
						$fileLibObject->date_creation = dol_now();
						$fileLibObject->fk_user_creat = $user->id;
						$resCreate = $fileLibObject->create($user);

						if (!$resCreate) {
							$this->resprints = "Failed to create history filelib for'".dol_basename($destfile)."'";
							return -1;
						}
						
					}

			}else{
				return 0;
			}

			return 0;

		}

		return 0;

	}
}
