<?php
/* Copyright (C) 2015   Jean-François Ferry     <jfefe@aternatik.fr>
 * Copyright (C) ---Put here your own copyright and developer email---
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

use Luracast\Restler\RestException;

require_once DOL_DOCUMENT_ROOT.'/salaries/class/salary.class.php';
require_once DOL_DOCUMENT_ROOT.'/salaries/class/paymentsalary.class.php';
require_once DOL_DOCUMENT_ROOT.'/loan/class/paymentloan.class.php';
require_once DOL_DOCUMENT_ROOT.'/compta/sociales/class/chargesociales.class.php';
require_once DOL_DOCUMENT_ROOT.'/compta/sociales/class/paymentsocialcontribution.class.php';
require_once DOL_DOCUMENT_ROOT.'/expensereport/class/paymentexpensereport.class.php';
require_once DOL_DOCUMENT_ROOT.'/expensereport/class/expensereport.class.php';
require_once DOL_DOCUMENT_ROOT.'/compta/tva/class/tva.class.php';

require_once DOL_DOCUMENT_ROOT.'/compta/tva/class/paymentvat.class.php';
require_once DOL_DOCUMENT_ROOT.'/core/lib/files.lib.php';
require_once DOL_DOCUMENT_ROOT.'/compta/bank/class/account.class.php';
require_once DOL_DOCUMENT_ROOT.'/fourn/class/paiementfourn.class.php';
require_once DOL_DOCUMENT_ROOT.'/compta/bank/class/paymentvarious.class.php';

require_once DOL_DOCUMENT_ROOT.'/custom/smartbankimport/core/modules/modSmartBankImport.class.php';
require_once DOL_DOCUMENT_ROOT.'/custom/smartbankimport/class/transactions.class.php';
require_once DOL_DOCUMENT_ROOT.'/custom/smartbankimport/class/fileslib.class.php';
require_once DOL_DOCUMENT_ROOT.'/custom/smartbankimport/lib/smartbankimport.lib.php';


/**
 * API class for Smart Bank Import
 *
 * @access protected
 * @class DolibarrApiAccess {@requires user,external}
 */
class SmartBankImportApi extends DolibarrApi
{

	/**
	 * Constructor
	 *
	 * @url     GET /
	 *
	 */
	public function __construct()
	{
		global $db;
		$this->db = $db;
	}


	/**
	 * Get the minimum information of company (public logo name and modules enables)
	 *
	 *
	 * @url     GET company
	 *
	 * @return array [List of main datas]
	 *
	 * @throws	RestException	400		Bad value for sqlfilters
	 * @throws	RestException	403		Access denied
	 * @throws	RestException	503		Error retrieving list of charge social types
	 */
	public function getCompanyMinimalInfos()
	{

		global $mysoc, $conf, $dolibarr_main_prod;

		if(isset($conf->modules)){
			sort($conf->modules);
		}

		$moduleSbi = new modSmartBankImport($this->db);
		
		return array(
			"logo_small" => isset($mysoc->logo_small) ? $mysoc->logo_small : null,
			"logo_squarred_small" => isset($mysoc->logo_squarred_small) ? $mysoc->logo_squarred_small : null,
			"name" =>  $mysoc->name,
			"modules" => $conf->modules,
			"dolibarr_main_prod" => $dolibarr_main_prod,
			"dol_version" => DOL_VERSION,
			"sbi_module_version" => $moduleSbi->version,
			"SMARTBANKIMPORT_ENABLEBANKINGENABLE" => $conf->global->SMARTBANKIMPORT_ENABLEBANKINGENABLE,
		);
		
	}

	/**
	 * Get Enable banking app ID and secret
	 *
	 *
	 * @url     GET enablebankingToken
	 *
	 * @return string JWT token string
	 *
	 * @throws	RestException	400		Bad value for sqlfilters
	 * @throws	RestException	403		Access denied
	 * @throws	RestException	503		Error retrieving list of charge social types
	 */
	public function getEnableBankingJwtToken()
	{
		
		global $conf;

		if (!DolibarrApiAccess::$user->hasRight('smartbankimport', 'write')) {
			throw new RestException(403);
		}

		return getEnableBankingJwtString($conf->global->SMARTBANKIMPORT_ENABLEBANKINGPEM, $conf->global->SMARTBANKIMPORT_ENABLEBANKINGAPPID);
		
	}

	
	/**
	 * Get balance for bank account
	 *
	 * @url     GET bankaccounts/{id}/balance
	 *
	 * @param	int			$id				ID of account
	 * @return  Object						Object with cleaned properties
	 *
	 * @throws RestException
	 */
	public function getBankAccountBalance($id)
	{
		if (!DolibarrApiAccess::$user->hasRight('banque', 'lire')) {
			throw new RestException(403);
		}

		$account = new Account($this->db);
		$result = $account->fetch($id);
		if (!$result) {
			throw new RestException(404, 'account not found');
		}

		return $account->solde(1);

	}

	/**
	 * Get list of unkinkede payment (no fk_bank)
	 *
	 * @url     GET unlinkedpayments
	 *
	 * @param	string			$paymenttype				paymenttype
	 * @return  Object						Object with cleaned properties
	 *
	 * @throws RestException
	 */
	public function getUnlinkedPayments($paymenttype)
	{
		if (!DolibarrApiAccess::$user->hasRight('banque', 'lire')) {
			throw new RestException(403);
		}

		if (empty($paymenttype)) {
			throw new RestException(403);
		}

		$entity = !empty(DolibarrApiAccess::$user->entity) ? DolibarrApiAccess::$user->entity : 1;

		//GET BANK ID LINE
		//---------------------------------------------------------------------------------------------------------------------------------
		if($paymenttype == "payment_supplier"){
			$sql = "SELECT ref, datep, amount, fk_paiement, note";
			$sql .= " FROM ".MAIN_DB_PREFIX."paiementfourn";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";
			$sql .= " AND entity = ".$entity;

		}else if($paymenttype == "payment"){
			$sql = "SELECT ref, datep, amount, fk_paiement, note";
			$sql .= " FROM ".MAIN_DB_PREFIX."paiement";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";
			$sql .= " AND entity = ".$entity;

		}else if($paymenttype == "paiementcharge"){
			$sql = "SELECT rowid as ref, datep, amount, fk_typepaiement as fk_paiement, note";
			$sql .= " FROM ".MAIN_DB_PREFIX."paiementcharge";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";
			//$sql .= " AND entity = ".$entity;

		}else if($paymenttype == "payment_loan"){
			$sql = "SELECT rowid as ref, datep, (amount_capital + amount_insurance + amount_interest) as amount, fk_typepayment as fk_paiement, note_private as note";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_loan";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";
			//$sql .= " AND entity = ".$entity;

		}else if($paymenttype == "payment_vat"){
			$sql = "SELECT rowid as ref, datep, amount, fk_typepaiement as fk_paiement, note";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_vat";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";
			//$sql .= " AND entity = ".$entity;

		}else if($paymenttype == "payment_expensereport"){
			$sql = "SELECT rowid as ref, datep, amount, fk_typepayment as fk_paiement, note";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_expensereport";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";
			//$sql .= " AND entity = ".$entity;

		}else if($paymenttype == "payment_salary"){
			$sql = "SELECT rowid as ref, datep, amount, fk_typepayment as fk_paiement, note";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_salary";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";
			$sql .= " AND entity = ".$entity;

		}else if($paymenttype == "payment_various"){
			$sql = "SELECT rowid as ref, datep, amount, fk_typepayment as fk_paiement, label as note";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_various";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";
			$sql .= " AND entity = ".$entity;

		}else{
			throw new RestException(404, "Unknown payment type : " . $paymenttype);
			return -1;
		}

		$result = $this->db->query($sql);

		if (empty($result)) {
			throw new RestException(503, $this->db->lasterror());
			return -1;
		}

		$list = array();
		$limit = 500;

		$num = $this->db->num_rows($result);

		$min = min($num, ($limit <= 0 ? $num : $limit));
		for ($i = 0; $i < $min; $i++) {
			$list[] = $this->db->fetch_object($result);
		}

		return $list;


	}

	

	/**
	 * Get the list of social contribution type
	 *
	 * @param string    $sortfield  Sort field
	 * @param string    $sortorder  Sort order
	 * @param int       $limit      Number of items per page
	 * @param int       $page       Page number {@min 0}
	 * @param int       $active     Charge social type is active or not {@min 0} {@max 1}
	 * @param string    $sqlfilters SQL criteria to filter with. Syntax example "(c.code:=:'TAXATAK')"
	 *
	 * @url     GET dictionary/chargesociales
	 *
	 * @return array [List of chargesociales types]
	 *
	 * @throws	RestException	400		Bad value for sqlfilters
	 * @throws	RestException	403		Access denied
	 * @throws	RestException	503		Error retrieving list of charge social types
	 */
	public function getSocialContributionTypes($sortfield = "libelle", $sortorder = 'ASC', $limit = 100, $page = 0, $active = 1, $sqlfilters = '')
	{
		global $mysoc;

		$list = array();

		if (!DolibarrApiAccess::$user->hasRight('tax', 'charges', 'lire')) {
			throw new RestException(403);
		}


		if (!empty($mysoc->country_id)) {
			$sql = "SELECT c.id, c.code, c.libelle as type";
			$sql .= " FROM ".MAIN_DB_PREFIX."c_chargesociales as c";
			$sql .= " WHERE c.active = ".((int) $active);
			$sql .= " AND c.fk_pays = ".((int) $mysoc->country_id);
			//$sql .= " AND entity = ".$entity;
			
		} else {
			$sql = "SELECT c.id, c.code, c.libelle as type";
			$sql .= " FROM ".MAIN_DB_PREFIX."c_chargesociales as c, ".MAIN_DB_PREFIX."c_country as co";
			$sql .= " WHERE c.active = ".((int) $active)." AND c.fk_pays = co.rowid";
			$sql .= " AND co.code = '".$this->db->escape($mysoc->country_code)."'";
			//$sql .= " AND entity = ".$entity;
		}

		if ($sqlfilters) {
			$errormessage = '';
			$sql .= forgeSQLFromUniversalSearchCriteria($sqlfilters, $errormessage);
			if ($errormessage) {
				throw new RestException(400, 'Error when validating parameter sqlfilters -> '.$errormessage);
			}
		}

		$sql .= $this->db->order($sortfield, $sortorder);

		if ($limit) {
			if ($page < 0) {
				$page = 0;
			}
			$offset = $limit * $page;

			$sql .= $this->db->plimit($limit, $offset);
		}

		$result = $this->db->query($sql);

		if ($result) {
			$num = $this->db->num_rows($result);
			$min = min($num, ($limit <= 0 ? $num : $limit));
			for ($i = 0; $i < $min; $i++) {
				$list[] = $this->db->fetch_object($result);
			}
		} else {
			throw new RestException(503, $this->db->lasterror());
		}

		return $list;
	}


	/**
	 * Get the list TAX / VAT values
	 *
	 * @param string    $sortfield  Sort field
	 * @param string    $sortorder  Sort order
	 * @param int       $limit      Number of items per page
	 * @param int       $page       Page number {@min 0}
	 * @param int       $active     Tax VAT is active or not {@min 0} {@max 1}
	 * @param int    	$country  	null return vat from mysoc country, provide country id
	 * @param string    $sqlfilters SQL criteria to filter with. Syntax example "(c.type_vat:=:0)"
	 *
	 * @url     GET dictionary/tva
	 *
	 * @return array [List of tva values]
	 *
	 * @throws	RestException	400		Bad value for sqlfilters
	 * @throws	RestException	403		Access denied
	 * @throws	RestException	503		Error retrieving list of charge social types
	 */
	public function getTva($sortfield = "taux", $sortorder = 'ASC', $limit = 100, $page = 0, $active = 1, $country = null, $sqlfilters = '')
	{
		global $mysoc;

		$list = array();

		if (!DolibarrApiAccess::$user->hasRight('tax', 'charges', 'lire')) {
			throw new RestException(403);
		}

		$entity = !empty(DolibarrApiAccess::$user->entity) ? DolibarrApiAccess::$user->entity : 1;

		$sql = "SELECT c.rowid, c.taux, c.note";
		$sql .= " FROM ".MAIN_DB_PREFIX."c_tva as c";
		$sql .= " WHERE c.active = ".((int) $active);

		if(empty($country)){
			$sql .= " AND c.fk_pays = ".((int) $mysoc->country_id);
		}else{
			$sql .= " AND c.fk_pays = ".((int) $country);
		}

		$sql .= " AND c.entity = ".$entity;
		

		if ($sqlfilters) {
			$errormessage = '';
			$sql .= forgeSQLFromUniversalSearchCriteria($sqlfilters, $errormessage);
			if ($errormessage) {
				throw new RestException(400, 'Error when validating parameter sqlfilters -> '.$errormessage);
			}
		}

		$sql .= $this->db->order($sortfield, $sortorder);

		if ($limit) {
			if ($page < 0) {
				$page = 0;
			}
			$offset = $limit * $page;

			$sql .= $this->db->plimit($limit, $offset);
		}

		$result = $this->db->query($sql);

		if ($result) {
			$num = $this->db->num_rows($result);
			$min = min($num, ($limit <= 0 ? $num : $limit));
			for ($i = 0; $i < $min; $i++) {
				$list[] = $this->db->fetch_object($result);
			}
		} else {
			throw new RestException(503, $this->db->lasterror());
		}

		return $list;
	}

	/**
	 * Get the list of loans
	 *
	 * @param string    $sortfield  Sort field
	 * @param string    $sortorder  Sort order
	 * @param int       $limit      Number of items per page
	 * @param int       $page       Page number {@min 0}
	 * @param int       $active     Loan is active or not {@min 0} {@max 1}
	 * @param string    $sqlfilters SQL criteria to filter with. Syntax example "(c.type_vat:=:0)"
	 *
	 * @url     GET loan
	 *
	 * @return array [List of tva values]
	 *
	 * @throws	RestException	400		Bad value for sqlfilters
	 * @throws	RestException	403		Access denied
	 * @throws	RestException	503		Error retrieving list of charge social types
	 */
	public function getLoans($sortfield = "rowid", $sortorder = 'ASC', $limit = 100, $page = 0, $active = 1, $sqlfilters = '')
	{
		global $mysoc;

		$list = array();

		if (!DolibarrApiAccess::$user->hasRight('loan', 'read')) {
			throw new RestException(403);
		}

		$entity = !empty(DolibarrApiAccess::$user->entity) ? DolibarrApiAccess::$user->entity : 1;

		$sql = "SELECT c.rowid, c.label, c.fk_bank, c.capital, c.datestart, c.dateend, c.rate, c.paid, c.fk_projet, c.active, c.insurance_amount";
		$sql .= " FROM ".MAIN_DB_PREFIX."loan as c";
		//$sql .= " LEFT JOIN ".MAIN_DB_PREFIX."loan_schedule as ls on (ls.fk_loan = c.rowid)";
		$sql .= " WHERE c.active = ".((int) $active);

		$sql .= " AND c.entity = ".$entity;
		

		if ($sqlfilters) {
			$errormessage = '';
			$sql .= forgeSQLFromUniversalSearchCriteria($sqlfilters, $errormessage);
			if ($errormessage) {
				throw new RestException(400, 'Error when validating parameter sqlfilters -> '.$errormessage);
			}
		}

		$sql .= $this->db->order($sortfield, $sortorder);

		if ($limit) {
			if ($page < 0) {
				$page = 0;
			}
			$offset = $limit * $page;

			$sql .= $this->db->plimit($limit, $offset);
		}

		$result = $this->db->query($sql);

		if ($result) {
			$num = $this->db->num_rows($result);
			$min = min($num, ($limit <= 0 ? $num : $limit));
			for ($i = 0; $i < $min; $i++) {
				$list[] = $this->db->fetch_object($result);
			}
		} else {
			throw new RestException(503, $this->db->lasterror());
		}

		return $list;
	}

	/**
	 * Create payment on loan
	 *
	 * @param 	int		$id					Id of loan
	 * @param 	array 	$request_data    	Request data
	 * @return 	int 						ID of payment_loan
	 *
	 * @url     POST loan/{id}/payments
	 *
	 * @throws RestException
	 */
	public function addPaymentLoan($id, $request_data = null)
	{

		if (!DolibarrApiAccess::$user->hasRight('loan', 'write')) {
			throw new RestException(403);
		}

		$paymentLoan = new PaymentLoan($this->db);

		$paymentLoan->chid = $id;

		foreach ($request_data as $field => $value) {
			$paymentLoan->$field = $this->_checkValForAPI($field, $value, $paymentLoan);
		}

		if ($paymentLoan->create(DolibarrApiAccess::$user, 1) < 0) {
			throw new RestException(500, 'Error creating payment_loan', array_merge(array($paymentLoan->error), $paymentLoan->errors));
		}

		if (isModEnabled("bank")) {
			$resPaymentToBank = $paymentLoan->addPaymentToBank(
				DolibarrApiAccess::$user,
				$id,
				'payment_loan',
				'(LoanPayment)',
				(int) $request_data['accountid'],
				'',
				''
			);

			if($resPaymentToBank < 0){
				throw new RestException(500, 'Error adding loan to bank : '.$paymentLoan->error);
				return -1;
			}
		}
		return $paymentLoan->id;
	}


	/**
	 * Transfer a file from library to transaction folder
	 *
	 *
	 * @param   string  $filename           	Name of file to create ('FA1705-0123.txt')
	 * @param   string  $subdir       			Subdirectory (Only if ref not provided)
	 * @param   int 	$overwriteifexists  	Overwrite file if exists (1 by default)
	 * @param   int 	$createdirifnotexists  	Create subdirectories if the doesn't exists (1 by default)
	 * @return  string
	 *
	 * @url PUT /transferfilefromlib
	 *
	 * @throws	RestException	400		Bad Request
	 * @throws	RestException	403		Access denied
	 * @throws	RestException	404		Object not found
	 * @throws	RestException	500		Error on file operationw
	 */
	public function transferfilefromlib($filename, $subdir = '', $overwriteifexists = 0, $createdirifnotexists = 1)
	{
		global $conf;

		$original_file = dol_sanitizeFileName($filename);

		if (!DolibarrApiAccess::$user->hasRight('smartbankimport', 'write')) {
			throw new RestException(403);
		}

		// Define $uploadir
		$object = null;

		$entity = !empty(DolibarrApiAccess::$user->entity) ? DolibarrApiAccess::$user->entity : 1;

		//GET DESTINATION FILE
		if(empty($subdir)){
			throw new RestException(400, 'bad value for parameter subdir');
			return -1;
		}

		//GET ORIGINAL FILE FROM LIBRARY
		$origine_filepath = $conf->smartbankimport->multidir_output[$entity]."/lib/".$original_file;

		if(!dol_is_file($origine_filepath)){
			throw new RestException(404, 'Original file not found : '.$origine_filepath);
			return -1;
		}

		$upload_dir = $conf->smartbankimport->multidir_output[$entity].'/'.$subdir;
		
		$upload_dir = dol_sanitizePathName($upload_dir);

		if (!empty($createdirifnotexists)) {
			if (dol_mkdir($upload_dir) < 0) { // needed by products
				throw new RestException(500, 'Error while trying to create directory '.$upload_dir);
			}
		}

		$destfile = $upload_dir.'/'.$original_file;
		$destfiletmp = DOL_DATA_ROOT.'/admin/temp/'.$original_file;
		dol_delete_file($destfiletmp);
		//var_dump($original_file);exit;

		if (!dol_is_dir(dirname($destfile))) {
			throw new RestException(400, 'Directory does not exists : '.dirname($destfile));
		}

		if (!$overwriteifexists && dol_is_file($destfile)) {
			throw new RestException(400, "File with name '".$original_file."' already exists.");
		}

		// in case temporary directory admin/temp doesn't exist
		if (!dol_is_dir(dirname($destfiletmp))) {
			dol_mkdir(dirname($destfiletmp));
		}

		$moreinfo = array('note_private' => 'File transfered with Smart Bank Import ');

		if (!empty($object) && is_object($object) && $object->id > 0) {
			$moreinfo['src_object_type'] = $object->table_element;
			$moreinfo['src_object_id'] = $object->id;
		}

		// Move the temporary file at its final emplacement
		$result = dol_move($origine_filepath, $destfile, 0, $overwriteifexists, 1, 1, $moreinfo);
		if (!$result) {
			throw new RestException(500, "Failed to move file into '".$destfile."'");
		}

		return dol_basename($destfile);
	}

	/**
	 * Create payment various
	 *
	 * @param 	array 	$request_data    	Request data
	 * @return 	int 						ID of payment_loan
	 *
	 * @url     POST paymentvarious
	 *
	 * @throws RestException
	 */
	public function createPaymentVarious($request_data = null)
	{

		if (!DolibarrApiAccess::$user->hasRight('banque', 'modifier')) {
			throw new RestException(403);
		}

		if(empty($request_data["fk_account"])){
			throw new RestException(404, "fk_account is required");
			return -1;
		}

		if(empty($request_data["datep"])){
			throw new RestException(404, "datep is required");
			return -1;
		}

		if(empty($request_data["datev"])){
			throw new RestException(404, "datev is required");
			return -1;
		}

		if($request_data["sens"] != 0 && $request_data["sens"] != 1 ){
			throw new RestException(404, "sens is required");
			return -1;
		}

		if(empty($request_data["note"])){
			throw new RestException(404, "note is required");
			return -1;
		}

		if(empty($request_data["type_payment"])){
			throw new RestException(404, "type_payment is required");
			return -1;
		}

		if(empty($request_data["label"])){
			throw new RestException(404, "label is required");
			return -1;
		}

		if(empty($request_data["amount"])){
			throw new RestException(404, "amount is required");
			return -1;
		}

		$object = new PaymentVarious($this->db);

		foreach ($request_data as $field => $value) {
			$object->$field = $this->_checkValForAPI($field, $value, $object);
		}

		$resCreate = $object->create(DolibarrApiAccess::$user, 1);

		if ($resCreate < 0) {
			throw new RestException(500, 'Error creating payment_various', array_merge(array($object->error), $object->errors));
		}

		return $resCreate;
	}

	/**
	 * Create payment of ExpenseReport
	 *
	 * @param 	int 	$id   							ID of expense report
	 * @param 	array 	$request_data   {@from body}  	Request data
	 * @return 	int 									ID of paymentExpenseReport
	 *
	 * @url     POST expenseReports/{id}/payments
	 */
	public function addPaymentExpenseReport($id, $request_data = null)
	{
		if (!DolibarrApiAccess::$user->hasRight('expensereport', 'creer')) {
			throw new RestException(403);
		}

		if(empty($request_data["fk_bank"])){
			throw new RestException(404, "fk_bank is required");
			return -1;
		}

		if(empty($request_data["amount"])){
			throw new RestException(404, "fk_bank is required");
			return -1;
		}

		if(empty($request_data["datep"])){
			throw new RestException(404, "fk_bank is required");
			return -1;
		}


		$paymentExpenseReport = new PaymentExpenseReport($this->db);
		$paymentExpenseReport->fk_expensereport = $id;
		foreach ($request_data as $field => $value) {
			$paymentExpenseReport->$field = $this->_checkValForAPI($field, $value, $paymentExpenseReport);
		}

		$resCreate = $paymentExpenseReport->create(DolibarrApiAccess::$user);

		if ($resCreate < 0) {
			throw new RestException(500, 'Error creating paymentExpenseReport', array_merge(array($paymentExpenseReport->error), $paymentExpenseReport->errors));
		}

		//SET EXPENSE REPORT TO PAID IF AMOUNT PAYMENT == total_ttc
		$expenseReport = new ExpenseReport($this->db);
		$resFetch = $expenseReport->fetch($id);

		if($resFetch > 0){
			if($request_data["amount"] == $expenseReport->total_ttc){
				$expenseReport->setPaid($id, DolibarrApiAccess::$user);
			}
		}

		//ADD PAYMENT TO BANK
		$paymentExpenseReport->fk_typepayment = $request_data["fk_typepayment"];
		$paymentExpenseReport->datep = $request_data["datep"];

		if (isModEnabled("bank")) {
			$paymentExpenseReport->addPaymentToBank(
				DolibarrApiAccess::$user,
				'payment_expensereport',
				'(ExpenseReportPayment)',
				(int) $request_data['fk_bank'],
				'',
				''
			);
		}

		return $resCreate;
	}

	/**
	 * Create and pay vat
	 *
	 * @param 	array 	$request_data    	Request data
	 * @return 	int 						ID of payment_loan
	 *
	 * @url     POST vat
	 *
	 * @throws RestException
	 */
	public function createPayVat($id, $request_data = null)
	{

		if (!DolibarrApiAccess::$user->hasRight('tax', 'charges', 'creer')) {
			throw new RestException(403);
		}

		$Tva = new Tva($this->db);

		foreach ($request_data as $field => $value) {
			$Tva->$field = $this->_checkValForAPI($field, $value, $Tva);
		}

		if ($Tva->create(DolibarrApiAccess::$user, 1) < 0) {
			throw new RestException(500, 'Error creating tva', array_merge(array($Tva->error), $Tva->errors));
			return false;
		}


		// Create a line of payments
		$paiement = new PaymentVAT($this->db);

		$paiement->chid         = $Tva->id;
		$paiement->datepaye     = $Tva->datep;
		$paiement->amounts      = array($Tva->id => $Tva->amount); // Tableau de montant
		$paiement->paiementtype = $Tva->type_payment;
		//$paiement->num_payment  = GETPOST("num_payment", 'alphanohtml');
		//$paiement->note = GETPOST("note", 'restricthtml');
		$paiement->note_private = $Tva->note_private;

		$closePaidTva = 1;
		$paymentid = $paiement->create(DolibarrApiAccess::$user, $closePaidTva);

		if ($paymentid < 0) {
			throw new RestException(500, 'Error creating paymenttva', array_merge(array($paiement->error), $paiement->errors));
			return false;
		}

		if (isModEnabled("bank")) {
			$result = $paiement->addPaymentToBank(DolibarrApiAccess::$user, 'payment_vat', '(VATPayment)', $Tva->fk_account, '', '');

			if (!($result > 0)) {
				throw new RestException(500, 'Error adding payment to bank', array_merge(array($paiement->error), $paiement->errors));
				return false;
			}
		}

		return $paiement->id;
	}

	/**
	 * Get the list of lines of the account (with link).
	 *
	 * @param int $id ID of account
	 * @return array Array of AccountLine objects
	 *
	 * @throws RestException
	 *
	 * @url GET bankaccounts/{id}/lines
	 * @param string    $sqlfilters Other criteria to filter answers separated by a comma. Syntax example "(t.ref:like:'SO-%') and (t.import_key:<:'20160101')"
	 */
	public function getLinesBankWithLink($id, $sqlfilters = '')
	{
		$list = array();

		if (!DolibarrApiAccess::$user->hasRight('banque', 'lire')) {
			throw new RestException(403);
		}

		$account = new Account($this->db);
		$result = $account->fetch($id);
		if (!$result) {
			throw new RestException(404, 'account not found');
		}


		$sql = "SELECT b.rowid, b.amount, b.datev, b.dateo, b.label, b.rappro, b.num_releve, b.fk_type, bl.url, bl.url_id, bl.type, bl.label AS labellink";
		$sql .= " FROM " . MAIN_DB_PREFIX . "bank_url AS bl";
		$sql .= " LEFT JOIN ".MAIN_DB_PREFIX."bank AS b ON (bl.fk_bank = b.rowid)";
		$sql .= " WHERE fk_account = " . ((int) $id);

		// Add sql filters
		if ($sqlfilters) {
			$errormessage = '';
			$sql .= forgeSQLFromUniversalSearchCriteria($sqlfilters, $errormessage);
			if ($errormessage) {
				throw new RestException(400, 'Error when validating parameter sqlfilters -> '.$errormessage);
			}
		}

		$sql .= " GROUP BY b.rowid";
		$sql .= " ORDER BY rowid";

		$result = $this->db->query($sql);

		$list = array();

		if ($result) {
			$num = $this->db->num_rows($result);
			for ($i = 0; $i < $num; $i++) {
				$obj = $this->db->fetch_object($result);

				$list[] = $obj;

				/*
				$accountLine = new AccountLine($this->db);
				if ($accountLine->fetch($obj->rowid) > 0) {
					$list[] = $this->_cleanObjectDatas($accountLine);
				}
				*/
			}
		} else {
			throw new RestException(503, 'Error when retrieving list of account lines: ' . $this->db->lasterror());
		}

		return $list;
	}


	/**
	 * Get transactions (Smart Bank Import)
	 *
	 * @param 	string    	$accountid   	gocardless account id
	 * @param 	string    	$start   		date start of fetch
	 * @param 	string    	$end		   	date end of fetch
	 * @param 	string    	$id_api_qonto		   	API QONTO ID
	 * @param 	string    	$key_api_qonto		   	API QONTO KEY
	 * 
	 * @return 	array		Object after update
	 *
	 * @throws RestException
	 *
	 * @url GET transactions
	 * 
	 */
	public function get_transactions($accountid, $start, $end = null)
	{
		global $conf;


		if (!DolibarrApiAccess::$user->hasRight('banque', 'lire')) {
			throw new RestException(403, "banque->lire right is off");
		}

		if (empty($accountid)) {
			throw new RestException(403, "accountid is required");
		}

		//CONVERT DATE FOR UNIVERSAL SEACH CRITERIA
		if(strlen($start) != 10){
			throw new RestException(400, "date start value is not correct");
			return -1;
		}
		if(!empty($end) && strlen($end) != 10){
			throw new RestException(400, "date end value is not correct");
			return -1;
		}

		$transactionObject = new TransactionsObj($this->db);

		$records = array();

		$limit = 1000;

		$sql = "SELECT *";
		$sql .= " FROM ".MAIN_DB_PREFIX."smartbankimport_transactions as t";
		$sql .= " WHERE fk_gocardless_bankaccount = '".$accountid."'";
		$sql .= " AND bookingdate >= '".$start."'";

		if(!empty($end)){
			$sql .= " AND bookingdate <= '".$end."'";
		}
		
		$sql .= " ORDER BY bookingdatetime DESC";
		$sql .= " LIMIT ".$limit;

		$resql = $this->db->query($sql);

		if ($resql) {
			$num = $this->db->num_rows($resql);
			$i = 0;

			//DIRLIST OF FILES
			$dirList = array();
			if(!empty($accountid)){
				///$upload_dir = "var/www/app.simple-soft.eu/documents/BANKIM/bankimport/7bc771aa-6215-4963-9f89-aea2db0c81e0";

				$upload_dir = dol_sanitizePathName($conf->smartbankimport->dir_output . "/" . $accountid);
				$utf8_path = $upload_dir;
				$types = "files";
				$recursive = 1;
				$filter = "";
				$excludefilter = ".json";
				$sortcriteria = "name";

				$dirList = dol_dir_list($utf8_path, $types, $recursive, $filter, $excludefilter, $sortcriteria);
			}


			while ($i < ($limit ? min($limit, $num) : $num)) {
				$obj = $this->db->fetch_object($resql);

				$record = new TransactionsObj($this->db);
				$record->setVarsFromFetchObj($obj);

				$fieldsToKeep = array_keys($transactionObject->fields);


				//DELETE UNECESSARY FIELDS
				foreach ($record as $key => $value) {
					if(!in_array($key,$fieldsToKeep) && $key != "id"){
						unset($record->$key);
					}
				}

				unset($record->fk_user_creat);
				unset($record->fk_user_modif);

				//GET FILES
				$record->files = array();
				foreach ($dirList as $value) {
					if($value["level1name"] == $record->id){
						$record->files[] = $value["name"];
					}
				}

				//SET LABEL
				if(!empty($record->creditorname)){
					$record->transactionlabel = $record->creditorname;

				}elseif (!empty($record->debtorname)) {
					$record->transactionlabel = $record->debtorname;

				}else{
					$record->transactionlabel = "";
				}
				

				//$records[$record->id] = $record;
				$records[] = $record;

				$i++;
			}
			$this->db->free($resql);

			return array(
				"datas" => $records,
				"balance" => null,
				"message" => ""
			);
		} else {

			throw new RestException(500, $this->db->lasterror());
			return -1;
		}
	}


	/**
	 * Add new TransactionsObj to database
	 *
	 * @param 	array 	$request_data    	Request data
	 * @return 	int 						ID of paymentsalary
	 *
	 * @url     POST transactions
	 *
	 * @throws RestException
	 */
	public function post_transactions($request_data = null)
	{

		if (!DolibarrApiAccess::$user->hasRight('banque', 'modifier')) {
			throw new RestException(403);
		}

		$this->db->begin();
		$error = 0;
		$transfered = 0;

		foreach ($request_data as $key => $value) {

			if(empty($value["internaltransactionid"])){
				$errorMsg = "internaltransactionid cannot be empty";
				$error++;
			}

			//CHECK FOR DOUBLONS
			$sql = "SELECT rowid, internaltransactionid";
			$sql .= " FROM ".MAIN_DB_PREFIX."smartbankimport_transactions as t";
			$sql .= " WHERE internaltransactionid = '".$value["internaltransactionid"]."'";
			$sql .= " AND fk_gocardless_bankaccount = ".$value["fk_gocardless_bankaccount"];

			$result = $this->db->query($sql);

			if ($result) {
				$num = $this->db->num_rows($result);

				if($num > 0){
					continue;
				}

			}else{
				throw new RestException(503, $this->db->lasterror());
			}

			//CREATE TRANSACTION
			$transactionObject = new TransactionsObj($this->db);

			$transactionObject->ref = $value["ref"];
			$transactionObject->date_creation = $value["date_creation"];
			$transactionObject->fk_user_creat = DolibarrApiAccess::$user->id;
			$transactionObject->fk_user_modif = DolibarrApiAccess::$user->id;
			$transactionObject->status = $value["status"];
			$transactionObject->bookingdate = $value["bookingdate"];
			$transactionObject->valuedate = $value["valuedate"];
			$transactionObject->bookingdatetime = $value["bookingdatetime"];
			$transactionObject->valuedatetime = $value["valuedatetime"];
			$transactionObject->transactionamount = $value["transactionamount"];
			$transactionObject->transactioncurrency = $value["transactioncurrency"];
			$transactionObject->creditorname = $value["creditorname"];
			$transactionObject->remittanceinformationunstru = $value["remittanceinformationunstru"];
			$transactionObject->fk_gocardless_bankaccount = $value["fk_gocardless_bankaccount"];
			$transactionObject->remittanceinformationjson = $value["remittanceinformationjson"];
			$transactionObject->additionalinformation = $value["additionalinformation"];
			$transactionObject->balanceafteramount = $value["balanceafteramount"];

			$transactionObject->balanceaftercurrency = $value["balanceaftercurrency"];
			$transactionObject->balanceaftertype = $value["balanceaftertype"];
			$transactionObject->internaltransactionid = $value["internaltransactionid"];
			$transactionObject->debtorname = $value["debtorname"];
			$transactionObject->debtoraccount_iban = $value["debtoraccount_iban"];
			$transactionObject->import_element_type = $value["import_element_type"];
			$transactionObject->import_element_id = $value["import_element_id"];
			$transactionObject->import_bankentry_id = $value["import_bankentry_id"];
			$transactionObject->import_datetime = $value["import_datetime"];
			$transactionObject->bankimport_id = $value["rowid"];


			$resUpdate = $transactionObject->create(DolibarrApiAccess::$user);

			if($resUpdate < 0){
				$errorMsg = "Could not create transactions : ".$this->db->lasterror();
				$error++;
			}else{
				$transfered++;
			}
		}

		
		if ($error > 0) {
			$this->db->rollback();

			//throw new RestException(500, $errorMsg);
			return -1 * $error;
		} else {
			$this->db->commit();
			return $transfered;
		}
	}


	/**
	 * Add new TransactionsObj to database AND COUNT DOUBLONS
	 *
	 * @param 	array 	$request_data    	Request data
	 * @return 	int 						ID of paymentsalary
	 *
	 * @url     POST transactionsdoublons
	 *
	 * @throws RestException
	 */
	public function post_transactionsdoublons($request_data = null)
	{

		if (!DolibarrApiAccess::$user->hasRight('banque', 'modifier')) {
			throw new RestException(403);
		}

		$this->db->begin();
		$error = 0;
		$transfered = 0;

		foreach ($request_data as $key => $value) {



			if(empty($value["internaltransactionid"])){
				$errorMsg = "internaltransactionid cannot be empty";
				$error++;
			}

			//CHECK FOR DOUBLONS
			$sql = "SELECT rowid, internaltransactionid";
			$sql .= " FROM ".MAIN_DB_PREFIX."smartbankimport_transactions as t";
			$sql .= " WHERE internaltransactionid = '".$value["internaltransactionid"]."'";

			$result = $this->db->query($sql);

			if ($result) {
				$num = $this->db->num_rows($result);

				if($num > 0){
					$transfered++;
					continue;
				}

			}else{
				throw new RestException(503, $this->db->lasterror());
			}

			//CREATE TRANSACTION
			$transactionObject = new TransactionsObj($this->db);

			$transactionObject->ref = $value["ref"];
			$transactionObject->date_creation = $value["date_creation"];
			$transactionObject->fk_user_creat = DolibarrApiAccess::$user->id;
			$transactionObject->fk_user_modif = DolibarrApiAccess::$user->id;
			$transactionObject->status = $value["status"];
			$transactionObject->bookingdate = $value["bookingdate"];
			$transactionObject->valuedate = $value["valuedate"];
			$transactionObject->bookingdatetime = $value["bookingdatetime"];
			$transactionObject->valuedatetime = $value["valuedatetime"];
			$transactionObject->transactionamount = $value["transactionamount"];
			$transactionObject->transactioncurrency = $value["transactioncurrency"];
			$transactionObject->creditorname = $value["creditorname"];
			$transactionObject->remittanceinformationunstru = $value["remittanceinformationunstru"];
			$transactionObject->fk_gocardless_bankaccount = $value["fk_gocardless_bankaccount"];
			$transactionObject->remittanceinformationjson = $value["remittanceinformationjson"];
			$transactionObject->additionalinformation = $value["additionalinformation"];
			$transactionObject->balanceafteramount = $value["balanceafteramount"];

			$transactionObject->balanceaftercurrency = $value["balanceaftercurrency"];
			$transactionObject->balanceaftertype = $value["balanceaftertype"];
			$transactionObject->internaltransactionid = $value["internaltransactionid"];
			$transactionObject->debtorname = $value["debtorname"];
			$transactionObject->debtoraccount_iban = $value["debtoraccount_iban"];
			$transactionObject->import_element_type = $value["import_element_type"];
			$transactionObject->import_element_id = $value["import_element_id"];
			$transactionObject->import_bankentry_id = $value["import_bankentry_id"];
			$transactionObject->import_datetime = $value["import_datetime"];
			$transactionObject->bankimport_id = $value["rowid"];


			$resUpdate = $transactionObject->create(DolibarrApiAccess::$user);

			if($resUpdate < 0){
				$errorMsg = "Could not create transactions : ".$this->db->lasterror();
				$error++;
			}else{
				$transfered++;
			}
		}

		
		if ($error) {
			$this->db->rollback();

			throw new RestException(500, $errorMsg);
			return -1 * $error;
		} else {
			$this->db->commit();
			return $transfered;
		}
	}


	/**
	 * Get the list of payments
	 *
	 * @param string    $elementtype	Type of payment (payment, payment_supplier, paiementcharge, payment_loan, payment_vat, payment_salary, payment_expensereport, payment_various)
	 * @param string    $sortfield  	Sort field
	 * @param string    $sortorder  	Sort order
	 * @param int       $limit      	Number of items per page
	 * @param int       $page       	Page number {@min 0}
	 * @param string    $sqlfilters 	SQL criteria to filter with. Syntax example "(p.rowid:=:'332')"
	 *
	 * @url     GET payments/{elementtype}
	 *
	 * @return array [List of chargesociales types]
	 *
	 * @throws	RestException	400		Bad value for sqlfilters
	 * @throws	RestException	403		Access denied
	 * @throws	RestException	503		Error retrieving list of charge social types
	 */
	public function getPayments($elementtype, $sortfield = "rowid", $sortorder = 'DESC', $limit = 100, $page = 0, $sqlfilters = '')
	{
		global $mysoc;

		$list = array();

		$entity = !empty(DolibarrApiAccess::$user->entity) ? DolibarrApiAccess::$user->entity : 1;

		if($elementtype == "payment"){		//customer invoices

			if (!DolibarrApiAccess::$user->hasRight("facture", "lire")) {
				throw new RestException(403);
			}

			$sql = "SELECT p.rowid, p.ref, p.datep, pf.fk_facture, p.amount, p.fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."paiement as p";
			$sql .= " LEFT JOIN ".MAIN_DB_PREFIX."paiement_facture as pf ON (p.rowid = pf.fk_paiement)";
			$sql .= " WHERE 1 = 1";

			$sql .= " AND p.entity = ".$entity;


		}elseif ($elementtype == "payment_supplier") {

			if (!DolibarrApiAccess::$user->hasRight("fournisseur", "facture", "lire")) {
				throw new RestException(403);
			}

			$sql = "SELECT p.rowid, p.ref, p.datep, pf.fk_facturefourn, pf.amount, p.fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."paiementfourn as p";
			$sql .= " LEFT JOIN ".MAIN_DB_PREFIX."paiementfourn_facturefourn as pf ON (p.rowid = pf.fk_paiementfourn)";
			$sql .= " WHERE 1 = 1";

			$sql .= " AND p.entity = ".$entity;


		}elseif ($elementtype == "paiementcharge") {

			if (!DolibarrApiAccess::$user->hasRight('tax', 'charges', 'lire')) {
				throw new RestException(403);
			}

			$sql = "SELECT p.rowid, p.rowid as ref, p.fk_charge, p.datep, p.amount, p.fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."paiementcharge as p";
			$sql .= " WHERE 1 = 1";

			//$sql .= " AND p.entity = ".$entity;

		}elseif ($elementtype == "payment_loan") {

			if (!DolibarrApiAccess::$user->hasRight('loan', 'write')) {
				throw new RestException(403);
			}

			$sql = "SELECT p.rowid, p.rowid as ref, p.fk_loan, p.datep, p.amount_capital, p.amount_insurance, p.amount_interest, p.fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_loan as p";
			$sql .= " WHERE 1 = 1";

			//$sql .= " AND p.entity = ".$entity;


		}elseif ($elementtype == "payment_vat") {

			if (!DolibarrApiAccess::$user->hasRight('tax', 'charges', 'lire')) {
				throw new RestException(403);
			}

			$sql = "SELECT p.rowid, p.rowid as ref, p.fk_tva, p.datep, p.amount, p.fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_vat as p";
			$sql .= " WHERE 1 = 1";

			//$sql .= " AND p.entity = ".$entity;

		}elseif ($elementtype == "payment_salary") {

			if (!DolibarrApiAccess::$user->hasRight('salaries', 'readall')) {
				throw new RestException(403);
			}

			$sql = "SELECT p.rowid, p.rowid as ref, p.fk_salary, p.datep, p.amount, p.fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_salary as p";
			$sql .= " WHERE 1 = 1";

			$sql .= " AND p.entity = ".$entity;

		}elseif ($elementtype == "payment_expensereport") {

			if (!DolibarrApiAccess::$user->hasRight('expensereport', 'readall')) {
				throw new RestException(403);
			}

			$sql = "SELECT p.rowid, p.rowid as ref, p.datep, p.fk_expensereport, p.amount, p.fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_expensereport as p";
			$sql .= " WHERE 1 = 1";

			//$sql .= " AND p.entity = ".$entity;
			

		}elseif ($elementtype == "payment_various") {

			if (!DolibarrApiAccess::$user->hasRight('banque', 'lire')) {
				throw new RestException(403);
			}

			$sql = "SELECT p.rowid, p.label as ref, p.datep, p.amount, p.sens, p.fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_various as p";
			$sql .= " WHERE 1 = 1";

			$sql .= " AND p.entity = ".$entity;


		}else{
			throw new RestException(403, 'Bad parameter for elementtype: '.$elementtype);
		}


		if ($sqlfilters) {
			$errormessage = '';
			$sql .= forgeSQLFromUniversalSearchCriteria($sqlfilters, $errormessage);
			if ($errormessage) {
				throw new RestException(400, 'Error when validating parameter sqlfilters -> '.$errormessage);
			}
		}

		if($elementtype == "payment" || $elementtype == "payment_supplier"){
			$sql .= " GROUP BY p.rowid";
		}

		$sql .= $this->db->order($sortfield, $sortorder);

		if ($limit) {
			if ($page < 0) {
				$page = 0;
			}
			$offset = $limit * $page;

			$sql .= $this->db->plimit($limit, $offset);
		}

		$result = $this->db->query($sql);

		if ($result) {
			$num = $this->db->num_rows($result);
			$min = min($num, ($limit <= 0 ? $num : $limit));
			for ($i = 0; $i < $min; $i++) {
				$list[] = $this->db->fetch_object($result);
			}
		} else {
			throw new RestException(503, $this->db->lasterror());
		}

		return $list;
	}

	/**
	 * Create payment salary on a salary
	 *
	 * @param 	int		$id					Id of salary
	 * @param 	array 	$request_data    	Request data
	 * @return 	int 						ID of paymentsalary
	 *
	 * @url     POST salaries/{id}/payments
	 *
	 * @throws RestException
	 */
	public function addPaymentSalaries($id, $request_data = null)
	{

		if (!DolibarrApiAccess::$user->hasRight('salaries', 'write')) {
			throw new RestException(403);
		}

		
		// Check mandatory fields
		//$result = $this->_validatepayments($request_data);

		$paymentsalary = new PaymentSalary($this->db);
		$paymentsalary->fk_salary = $id;
		foreach ($request_data as $field => $value) {
			$paymentsalary->$field = $this->_checkValForAPI($field, $value, $paymentsalary);
		}

		//Create payment
		$idPayment = $paymentsalary->create(DolibarrApiAccess::$user, 1);

		if ($idPayment < 0) {
			throw new RestException(500, 'Error creating paymentsalary', array_merge(array($paymentsalary->error), $paymentsalary->errors));
		}

		//FETCH PAYMENT SALARY
		$resFetch = $paymentsalary->fetch($idPayment);

		if ($resFetch < 0) {
			throw new RestException(500, 'Error fetching paymentsalary' . $paymentsalary->error);
			return -1;
		}

		//BE SURE THAT fk_typepayment is defined
		if(empty($paymentsalary->fk_typepayment) && !empty($paymentsalary->type_payment)){
			$paymentsalary->fk_typepayment = $paymentsalary->type_payment;

		}elseif(empty($paymentsalary->fk_typepayment) && !empty($request_data["type_payment"])){
			$paymentsalary->fk_typepayment = $request_data["type_payment"];

		}elseif(!empty($paymentsalary->fk_typepayment)){
			//ok continue
		}else{
			throw new RestException(500, 'Type payment should be defined (653)');
			return -1;
		}

		if (isModEnabled("bank")) {
			$resPaymentToBank = $paymentsalary->addPaymentToBank(
				DolibarrApiAccess::$user,
				'payment_salary',
				'(SalaryPayment)',
				(int) $request_data['accountid'],
				'',
				''
			);

			if($resPaymentToBank < 0){
				throw new RestException(500, 'Error adding salary to bank : '.$paymentsalary->error);
				return -1;
			}
		}

		return $idPayment;
	}


	/**
	 * Correction Dolibarr for empty fk_bank fields in payments
	 *
	 * @param 	array 	$request_data    	Request data
	 * @return 	int 						1 or -1
	 *
	 * @url     POST paymentstobank/correction
	 *
	 * @throws RestException
	 */
	public function paymentToBankCorrection($request_data)
	{

		if (!DolibarrApiAccess::$user->hasRight('banque', 'modifier')) {
			throw new RestException(403);
		}

		if (empty($request_data["accountid"])) {
			throw new RestException(404, "accountid is required");
			return -1;
		}

		if (empty($request_data["paymentmode"])) {
			throw new RestException(404, "paymentmode is required");
			return -1;
		}

		if (empty($request_data["type_payment"])) {
			throw new RestException(404, "type_payment is required");
			return -1;
		}

		$paymenttype = $request_data["type_payment"];


		//GET BANK ID LINE
		//---------------------------------------------------------------------------------------------------------------------------------
		/*
		if($paymenttype == "payment_supplier"){
			$sql = "SELECT rowid, ref, datep, amount, fk_paiement, note";
			$sql .= " FROM ".MAIN_DB_PREFIX."paiementfourn";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";

		}else if($paymenttype == "payment"){
			$sql = "SELECT rowid, ref, datep, amount, fk_paiement, note";
			$sql .= " FROM ".MAIN_DB_PREFIX."paiement";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";

		}else if($paymenttype == "payment_various"){
			$sql = "SELECT rowid, rowid as ref, datep, amount, fk_typepayment as fk_paiement, label as note";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_various";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";

		}else if($paymenttype == "payment_loan"){
			$sql = "SELECT rowid, rowid as ref, datep, (amount_capital + amount_insurance + amount_interest) as amount, fk_typepayment as fk_paiement, note_private as note";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_loan";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";

		}else if($paymenttype == "payment_vat"){
			$sql = "SELECT rowid, rowid as ref, datep, amount, fk_typepaiement as fk_paiement, note";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_vat";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";

			*/

		if($paymenttype == "paiementcharge"){
			$sql = "SELECT rowid, rowid as ref, datep, amount, fk_typepaiement as fk_paiement, note";
			$sql .= " FROM ".MAIN_DB_PREFIX."paiementcharge";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";

		}else if($paymenttype == "payment_expensereport"){
			$sql = "SELECT rowid, rowid as ref, datep, amount, fk_typepayment as fk_paiement, note";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_expensereport";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";

		}else if($paymenttype == "payment_salary"){
			$sql = "SELECT rowid, rowid as ref, datep, amount, fk_typepayment as fk_paiement, note";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_salary";
			$sql .= " WHERE fk_bank IS NULL OR fk_bank = '' OR fk_bank = 0";


		}else{
			throw new RestException(404, "Unknown payment type : " . $paymenttype);
			return -1;
		}

		$result = $this->db->query($sql);

		if (empty($result)) {
			throw new RestException(503, $this->db->lasterror());
			return -1;
		}


		$limit = 500;

		$num = $this->db->num_rows($result);

		$min = min($num, ($limit <= 0 ? $num : $limit));
		for ($i = 0; $i < $min; $i++) {

			$obj = $this->db->fetch_object($result);

			if($paymenttype == "paiementcharge"){
				$object = new PaymentSocialContribution($this->db);
							
				$resFetch = $object->fetch($obj->rowid);

				if ($resFetch < 0) {
					throw new RestException(500, 'Error fetching '.$paymenttype . ": " . $object->error);
					return -1;
				}

				$object->paiementtype = $obj->fk_paiement ? $obj->fk_paiement : $request_data["paymentmode"];
				$object->datepaye = $obj->datep;

				if (isModEnabled("bank")) {
					$resPaymentToBank = $object->addPaymentToBank(
						DolibarrApiAccess::$user,
						'payment_sc',
						'(SocialContributionPayment)',
						(int) $request_data['accountid'],
						'',
						''
					);

					if($resPaymentToBank < 0){
						throw new RestException(500, 'Error adding social charge to bank : '.$object->error);
						return -1;
					}
				}

			}else if($paymenttype == "payment_expensereport"){

				$object = new PaymentExpenseReport($this->db);
							
				$resFetch = $object->fetch($obj->rowid);

				if ($resFetch < 0) {
					throw new RestException(500, 'Error fetching '.$paymenttype . ": " . $object->error);
					return -1;
				}

				//$object->fk_typepayment = $request_data["fk_typepayment"];
				//$object->datep = $request_data["datep"];

				if (isModEnabled("bank")) {
					$resPaymentToBank = $object->addPaymentToBank(
						DolibarrApiAccess::$user,
						'payment_expensereport',
						'(ExpenseReportPayment)',
						(int) $request_data['accountid'],
						'',
						''
					);

					if($resPaymentToBank < 0){
						throw new RestException(500, 'Error adding expense report to bank : '.$object->error);
						return -1;
					}

				}else{
					throw new RestException(500, 'Module bank is disabled');
					return -1;
				}

			}else if($paymenttype == "payment_salary"){
				$object = new PaymentSalary($this->db);
							
				$resFetch = $object->fetch($obj->rowid);

				if ($resFetch < 0) {
					throw new RestException(500, 'Error fetching '.$paymenttype . ": " . $object->error);
					return -1;
				}

				$object->fk_bank = $request_data["accountid"];
				$object->fk_typepayment = $request_data["paymentmode"];

				$object->update(DolibarrApiAccess::$user);

				if (isModEnabled("bank")) {
					$resPaymentToBank = $object->addPaymentToBank(
						DolibarrApiAccess::$user,
						'payment_salary',
						'(SalaryPayment)',
						(int) $request_data['accountid'],
						'',
						''
					);

					if($resPaymentToBank < 0){
						throw new RestException(500, 'Error adding salary to bank : '.$object->error);
						return -1;
					}

				}else{
					throw new RestException(500, 'Module bank is disabled');
					return -1;
				}

			}else{
				throw new RestException(404, "Unknown payment type : " . $paymenttype);
				return -1;
			}

		}

		return 1;
	}

	/**
	 * Create social contribution
	 *
	 * @param 	array 	$request_data    	Request data (type, fk_account, mode_reglement_id, label, date_ech, periode, amount, fk_project, fk_user)
	 * @return 	int 						ID of chargesociales
	 *
	 * @url     POST chargesociales
	 *
	 * @throws RestException
	 */
	public function createSocialTax($request_data = null)
	{

		if (!DolibarrApiAccess::$user->hasRight('tax', 'charges', 'creer')) {
			throw new RestException(403);
		}

		$chargeSocialesObject = new ChargeSociales($this->db);

		foreach ($request_data as $field => $value) {
			$chargeSocialesObject->$field = $value;
		}

		if ($chargeSocialesObject->create(DolibarrApiAccess::$user, 1) < 0) {
			throw new RestException(500, 'Error creating paymentsalary', array_merge(array($chargeSocialesObject->error), $chargeSocialesObject->errors));
		}

		return $chargeSocialesObject->id;
	}

	/**
	 * Create social contribution. payment
	 *
	 * @param 	int		$id					Id of chargessociales
	 * @param 	array 	$request_data    	Request data (amounts, datepaye, paiementtype, note)
	 * @return 	int 						ID of paymentsalary
	 *
	 * @url     POST chargesociales/{id}/payments
	 *
	 * @throws RestException
	 */
	public function createSocialTaxPayment($id, $request_data = null)
	{

		if (!DolibarrApiAccess::$user->hasRight('tax', 'charges', 'creer')) {
			throw new RestException(403);
		}

		if (empty($id)) {
			throw new RestException(404, "id is required");
		}

		$chargeSocialesObject = new ChargeSociales($this->db);
		$resChargeSocial = $chargeSocialesObject->fetch($id);

		if($resChargeSocial < 0){
			throw new RestException(500, 'Error fetching social charge id:'.$id);
		}

		$paymentSocialContributionObject = new PaymentSocialContribution($this->db);
		$paymentSocialContributionObject->chid = $id;

		foreach ($request_data as $field => $value) {
			$paymentSocialContributionObject->$field = $value;
		}

		if ($paymentSocialContributionObject->create(DolibarrApiAccess::$user, 1) < 0) {
			throw new RestException(500, 'Error creating social tax', array_merge(array($paymentSocialContributionObject->error), $paymentSocialContributionObject->errors));
		}

		if (isModEnabled("bank")) {
			$resPaymentToBank = $paymentSocialContributionObject->addPaymentToBank(
				DolibarrApiAccess::$user,
				'payment_sc',
				'(SocialContributionPayment)',
				(int) $request_data['fk_account'],
				'',
				''
			);

			if($resPaymentToBank < 0){
				throw new RestException(500, 'Error adding social tax to bank : '.$paymentSocialContributionObject->error);
				return -1;
			}
		}

		return $paymentSocialContributionObject->id;
	}



	/**
	 * reconciliate from payment
	 *
	 * @param 	array 	$request_data    	Request data (idpayment, paymenttype, refstatement)
	 * @return 	int 						ID of paymentsalary
	 *
	 * @url     POST reconciliate
	 * 
	 *
	 * @throws RestException
	 */
	public function reconciliateFromPayment($request_data)
	{

		if (!DolibarrApiAccess::$user->hasRight('banque', 'consolidate')) {
			throw new RestException(403, "Dolibarr rights banque -> consolidate is needed");
			return -1;
		}


		if (empty($request_data["idpayment"]) && empty($request_data["identry"])) {
			throw new RestException(404, "idpayment or identry are required");
			return -1;
		}

		if (empty($request_data["paymenttype"])) {
			throw new RestException(404, "refstatement is required");
			return -1;
		}

		if(substr($request_data["refstatement"], 0, 2) != "20" && substr($request_data["refstatement"], 0, 4) != "SBI_"){	//accept SBI_YYYYMM or YYYYMM
			throw new RestException(404, "refstatement format error (749)");
			return -1;
		}

		if(strlen($request_data["refstatement"]) != 6 && strlen($request_data["refstatement"]) != 10){	//accept SBI_YYYYMM or YYYYMM
			throw new RestException(404, "refstatement format error (753)");
			return -1;
		}


		if ($request_data["paymenttype"] == "bankentry" && empty($request_data["identry"])) {
			throw new RestException(404, "identry is required for bankentry type");
			return -1;
		}

		$entity = !empty(DolibarrApiAccess::$user->entity) ? DolibarrApiAccess::$user->entity : 1;


		$sqlRequestNeeded = true;

		//GET BANK ID LINE
		//---------------------------------------------------------------------------------------------------------------------------------
		if($request_data["paymenttype"] == "payment_supplier"){
			$sql = "SELECT fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."paiementfourn";
			$sql .= " WHERE rowid = ".((int) $request_data["idpayment"]);

			$sql .= " AND entity = ".$entity;

		}else if($request_data["paymenttype"] == "payment"){
			$sql = "SELECT fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."paiement";
			$sql .= " WHERE rowid = ".((int) $request_data["idpayment"]);

			$sql .= " AND entity = ".$entity;

		}else if($request_data["paymenttype"] == "paiementcharge"){
			$sql = "SELECT fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."paiementcharge";
			$sql .= " WHERE rowid = ".((int) $request_data["idpayment"]);

			//$sql .= " AND entity = ".$entity;

		}else if($request_data["paymenttype"] == "payment_loan"){
			$sql = "SELECT fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_loan";
			$sql .= " WHERE rowid = ".((int) $request_data["idpayment"]);

			//$sql .= " AND entity = ".$entity;

		}else if($request_data["paymenttype"] == "payment_vat"){
			$sql = "SELECT fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_vat";
			$sql .= " WHERE rowid = ".((int) $request_data["idpayment"]);

			//$sql .= " AND entity = ".$entity;

		}else if($request_data["paymenttype"] == "payment_expensereport"){
			$sql = "SELECT fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_expensereport";
			$sql .= " WHERE rowid = ".((int) $request_data["idpayment"]);

			//$sql .= " AND entity = ".$entity;

		}else if($request_data["paymenttype"] == "payment_salary"){
			$sql = "SELECT fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_salary";
			$sql .= " WHERE rowid = ".((int) $request_data["idpayment"]);

			$sql .= " AND entity = ".$entity;

		}else if($request_data["paymenttype"] == "payment_various"){
			$sql = "SELECT fk_bank";
			$sql .= " FROM ".MAIN_DB_PREFIX."payment_various";
			$sql .= " WHERE rowid = ".((int) $request_data["idpayment"]);

			$sql .= " AND entity = ".$entity;

		}else if($request_data["paymenttype"] == "bankentry"){
			$sqlRequestNeeded = false;

		}else{
			throw new RestException(404, "Unknown payment type : " . $request_data["paymenttype"]);
			return -1;
		}

		if($sqlRequestNeeded){

			$result = $this->db->query($sql);

			if (empty($result)) {
				throw new RestException(503, $this->db->lasterror());
				return -1;
			}

			$paymentObject = $this->db->fetch_object($result);


			if(empty($paymentObject->fk_bank)){
				throw new RestException(500, "Bank line id could be found");
				return -1;
			}
		}


		// Conciliation
		//---------------------------------------------------------------------------------------------------------------------------------
		$bankline = new AccountLine($this->db);

		$lineEntryId = $request_data["paymenttype"] == "bankentry" ? $request_data["identry"] : $paymentObject->fk_bank;

		$result = $bankline->fetch($lineEntryId);
		$bankline->num_releve = $request_data["refstatement"];
		$conciliate = 1;
		$categorie = null;

		$result = $bankline->update_conciliation(DolibarrApiAccess::$user, $categorie, $conciliate);

		if ($result < 0) {
			throw new RestException(404, "error conciliation");
			return -1;

		}else{
			return 1;
		}
	}

	/**
	 * Update transactions import_element_type and import_element_id
	 *
	 * @param int   $id             Id of supplier invoice to update
	 * @param array $request_data   supplier invoice line data
	 *
	 * @url	PUT transactions/{id}/import
	 *
	 * @return int|bool
	 *
	 * @throws RestException 403
	 * @throws RestException 404
	 */
	public function put_importtransaction($id, $request_data = null)
	{
		//CHECK USER IS ALLOWED
		//---------------------------------------------------------------------------------------------------------------------
		//---------------------------------------------------------------------------------------------------------------------
		if (!DolibarrApiAccess::$user->hasRight('smartbankimport', 'write')) {
			throw new RestException(403);
		}

		if(empty($id) || $id < 0){
			throw new RestException("403", "error api 194");
			return -1;
		}

		if(!array_key_exists("import_element_type",$request_data)){
			throw new RestException("403", "error api 197");
			return -1;
		}

		if(!array_key_exists("import_element_id",$request_data)){
			throw new RestException("403", "error api 201");
			return -1;
		}

        $transactionsObject = new TransactionsObj($this->db);

		$resFetch = $transactionsObject->fetch($id);

		if($resFetch <= 0){
			throw new RestException("500", "(1722) ".$transactionsObject->error);
			return -1;
		}

		$transactionsObject->import_element_type = $request_data["import_element_type"];
		$transactionsObject->import_element_id = $request_data["import_element_id"];

		$result = $transactionsObject->update(DolibarrApiAccess::$user);

		if ($result < 0) {
			throw new RestException(500, $transactionsObject->error);
		}


		$bankEntryId = 0;
		
		//SI BANKENTRY, WE DONT NEED TO REQUEST DOLIBARR SERVER
		if(!empty($request_data["import_element_type"])){
			if($request_data["import_element_type"] == "bankentry"){
					
				$transactionsObject->import_bankentry_id = $transactionsObject->import_element_id;

				$resUpdate = $transactionsObject->update(DolibarrApiAccess::$user);

				if($resUpdate <= 0){
					throw new RestException(500, "(1742) error saving bank entry id : " . $transactionsObject->error);
					return -1;
				}

				$bankEntryId = $transactionsObject->import_element_id;

			}else{

				//WE REQUEST DOLIBARR SERVER TO KNOW fk_bank VALUE
				$requestResponse = $this->getPayments($request_data["import_element_type"], "rowid", 'DESC', 100, 0, '(p.rowid:=:'.$request_data["import_element_id"].')');


				if(is_array($requestResponse) && count ($requestResponse) == 1 && $requestResponse[0]->fk_bank > 0){

					$transactionsObject->import_bankentry_id = $requestResponse[0]->fk_bank;

					$resUpdate = $transactionsObject->update(DolibarrApiAccess::$user);

					if($resUpdate <= 0){
						throw new RestException(500, "error 1159 : " . $transactionsObject->error);
						return -1;
					}

					$bankEntryId = $requestResponse[0]->fk_bank;

				}else{


					//WE DON't THROW ERROR

					if(false){
						if(is_array($requestResponse) && count ($requestResponse) == 1){
							throw new RestException(500, "error 1166 empty fk_bank :". $request_data["import_element_type"] ." : ". $request_data["import_element_id"]);
							return -1;
						}else{
							throw new RestException(500, "error 1169 (error fetching payments)");
							return -1;
						}
					}

				}

			}
		}


		//RETURN
		if($bankEntryId > 0 && $result > 0){
			return $bankEntryId;

		}elseif(empty($bankEntryId) && $result > 0){
			return -1;

		}else{
			return -2;
		}

	}

	/**
	 * Download a document.
	 *
	 * Note that, this API is similar to using the wrapper link "documents.php" to download a file (used for
	 * internal HTML links of documents into application), but with no need to have a session cookie (the token is used instead).
	 *
	 * @param   string  $modulepart     Name of module or area concerned by file download ('facture', ...)
	 * @param   string  $original_file  Relative path with filename, relative to modulepart (for example: IN201701-999/IN201701-999.pdf)
	 * @return  array                   List of documents
	 *
	 * @url GET /downloadfile
	 *
	 * @throws	RestException	400		Bad value for parameter modulepart or original_file
	 * @throws	RestException	403		Access denied
	 * @throws	RestException	404		File not found
	 */
	public function downloadDocument($modulepart, $original_file = '')
	{
		global $conf;

		if (empty($modulepart) || $modulepart != "smartbankimport") {
			throw new RestException(400, 'bad value for parameter modulepart');
		}

		if (empty($original_file)) {
			throw new RestException(400, 'bad value for parameter original_file');
		}


		if (!DolibarrApiAccess::$user->hasRight('smartbankimport', 'write')) {
			throw new RestException(403);
		}

		$entity = !empty(DolibarrApiAccess::$user->entity) ? DolibarrApiAccess::$user->entity : 1;

		$original_file = $conf->smartbankimport->multidir_output[$entity].'/'.$original_file;

		if (preg_match('/\.\./', $original_file) || preg_match('/[<>|]/', $original_file)) {
			throw new RestException(403);
		}

		$filename = basename($original_file);
		$original_file_osencoded = dol_osencode($original_file); // New file name encoded in OS encoding charset


		if (!file_exists($original_file_osencoded)) {
			dol_syslog("Try to download not found file ".$original_file_osencoded, LOG_WARNING);
			throw new RestException(404, 'File not found');
		}

		$file_content = file_get_contents($original_file_osencoded);

		return array('filename'=>$filename, 'content-type' => dol_mimetype($filename), 'filesize'=>filesize($original_file), 'content'=>base64_encode($file_content), 'encoding'=>'base64');
	}

	/**
	 * Get history of files in library
	 *
	 *
	 * @return  array                   List of documents
	 *
	 * @url GET /historylibraryfiles
	 *
	 * @throws	RestException	400		Bad value for parameter modulepart or original_file
	 * @throws	RestException	403		Access denied
	 */
	public function historyFilesLib($sbiuser, $readall = 0)
	{

		if (!DolibarrApiAccess::$user->hasRight('smartbankimport', 'write')) {
			throw new RestException(403);
		}
	
		//GET LIST OF UPLOADED FILES
		//***************************************************************************************************************** */
		$sql = "SELECT fh.rowid, fh.label, fh.date_creation, fh.fk_user_creat";
		$sql .= " FROM ".MAIN_DB_PREFIX."smartbankimport_fileslib AS fh";

		if(!empty($readall)){
			$sql .= " WHERE 1 = 1";

		}else{
			$sql .= " WHERE fh.fk_user_creat = " . $sbiuser;

		}

		$sql .= " ORDER BY fh.date_creation DESC";
		$sql .= " LIMIT 200";
		
		$resql = $this->db->query($sql);

		if (!$resql) {
			$error = "(2635) error getting list of file";
			throw new RestException(500, $error);
			return -1;
		}
		
		$num = $this->db->num_rows($resql);

		$fileHistory = array();

		if ($num >= 1) {

			$i = 0;
			while ($i < $num) {
				$obj = $this->db->fetch_object($resql);

				$fileHistory[] = $obj;

				$i++;
			}
		}

		return $fileHistory;
		
	}


	/**
	 * Get list of library files
	 *
	 *
	 * @return  array                   List of documents
	 *
	 * @url GET /listoflibraryfiles
	 *
	 * @throws	RestException	400		Bad value for parameter modulepart or original_file
	 * @throws	RestException	403		Access denied
	 */
	public function listoflibraryfiles()
	{
		global $conf;

		//SECURITY : CHECK USER HAS RIGHT
		
		if(empty(DolibarrApiAccess::$user->hasRight("smartbankimport", "write"))){
			throw new RestException(403, 'User not allowed');
			return -1;
		}


		$entity = !empty(DolibarrApiAccess::$user->entity) ? DolibarrApiAccess::$user->entity : 1;


		//DIRLIST OF FILES
		$upload_dir = dol_sanitizePathName($conf->smartbankimport->multidir_output[$entity] . "/lib");
		$utf8_path = $upload_dir;
		$types = "files";
		$recursive = 1;
		$filter = "";
		$excludefilter = ".json";
		$sortcriteria = "name";

		return dol_dir_list($utf8_path, $types, $recursive, $filter, $excludefilter, $sortcriteria);
	}


	/**
	 * Upload a document.
	 *
	 * Test sample for invoice: { "filename": "mynewfile.txt", "modulepart": "invoice", "ref": "FA1701-001", "subdir": "", "filecontent": "content text", "fileencoding": "", "overwriteifexists": "0" }.
	 * Test sample for supplier invoice: { "filename": "mynewfile.txt", "modulepart": "supplier_invoice", "ref": "FA1701-001", "subdir": "", "filecontent": "content text", "fileencoding": "", "overwriteifexists": "0" }.
	 * Test sample for medias file: { "filename": "mynewfile.txt", "modulepart": "medias", "ref": "", "subdir": "image/mywebsite", "filecontent": "Y29udGVudCB0ZXh0Cg==", "fileencoding": "base64", "overwriteifexists": "0" }.
	 *
	 * Supported modules: invoice, order, supplier_order, task/project_task, product/service, expensereport, fichinter, member, propale, agenda, contact
	 *
	 * @param   string  $filename           	Name of file to create ('FA1705-0123.txt')
	 * @param   string  $modulepart         	Name of module or area concerned by file upload ('product', 'service', 'invoice', 'proposal', 'project', 'project_task', 'supplier_invoice', 'expensereport', 'member', ...)
	 * @param   string  $ref                	Reference of object (This will define subdir automatically and store submitted file into it)
	 * @param   string  $subdir       			Subdirectory (Only if ref not provided)
	 * @param   string  $filecontent        	File content (string with file content. An empty file will be created if this parameter is not provided)
	 * @param   string  $fileencoding       	File encoding (''=no encoding, 'base64'=Base 64)
	 * @param   int 	$overwriteifexists  	Overwrite file if exists (1 by default)
	 * @param   int 	$createdirifnotexists  	Create subdirectories if the doesn't exists (1 by default)
	 * @param   int 	$sbiuserid  			Smart Bank Import user ID
	 * 
	 * 
	 * @return  string
	 *
	 * @url POST /upload
	 *
	 * @throws	RestException	400		Bad Request
	 * @throws	RestException	403		Access denied
	 * @throws	RestException	404		Object not found
	 * @throws	RestException	500		Error on file operationw
	 */
	public function post_document($filename, $modulepart, $ref = '', $subdir = '', $filecontent = '', $fileencoding = '', $overwriteifexists = 0, $createdirifnotexists = 1, $sbiuserid = 0)
	{
		global $conf;

		//var_dump($modulepart);
		//var_dump($filename);
		//var_dump($filecontent);exit;

		$modulepartorig = $modulepart;

		if (empty($modulepart)) {
			throw new RestException(400, 'Modulepart not provided.');
		}

		$newfilecontent = '';
		if (empty($fileencoding)) {
			$newfilecontent = $filecontent;
		}
		if ($fileencoding == 'base64') {
			$newfilecontent = base64_decode($filecontent);
		}

		$original_file = dol_sanitizeFileName($filename);

		// Define $uploadir
		$object = null;

		$entity = !empty(DolibarrApiAccess::$user->entity) ? DolibarrApiAccess::$user->entity : 1;


		if ($modulepart == 'salaries') {

			if (empty($ref)) {
				throw new RestException(400, 'Ref not provided.');
			}

			if (!DolibarrApiAccess::$user->hasRight('salaries', 'write')) {
				throw new RestException(403);
			}

			$object = new Salary($this->db);
			$upload_dir = $conf->salaries->dir_output.'/'.$ref;
			$result = $object->fetch($ref);

		}elseif ($modulepart == 'tax_vat') {

			if (empty($ref)) {
				throw new RestException(400, 'Ref not provided.');
			}
		
			if (!DolibarrApiAccess::$user->hasRight('tax', 'charges', 'creer')) {
				throw new RestException(403);
			}

			$object = new Tva($this->db);
			$upload_dir = $conf->tax->dir_output.'/vat/'.$ref;
			$result = $object->fetch($ref);
			
		}elseif ($modulepart == 'smartbankimport') {
		
			if (!DolibarrApiAccess::$user->hasRight('smartbankimport', 'write')) {
				throw new RestException(403);
			}

			
			$upload_dir = $conf->smartbankimport->multidir_output[$entity].'/'.$subdir;

			$result = 1;

		} else {
			// TODO Implement additional moduleparts
			throw new RestException(500, 'Modulepart '.$modulepart.' not implemented yet.');
		}

		if ($result == 0) {
			throw new RestException(404, "Object with ref '".$ref."' was not found.");
		} elseif ($result < 0) {
			throw new RestException(500, 'Error while fetching object: '.$object->error);
		}

		if (!($object->id > 0) && $modulepart != 'smartbankimport') {
			throw new RestException(404, 'The object '.$modulepart." with ref '".$ref."' was not found.");
		}
		

		if (empty($upload_dir) || $upload_dir == '/') {
			throw new RestException(500, 'This value of modulepart ('.$modulepart.') does not support yet usage of ref. Check modulepart parameter or try to use subdir parameter instead of ref.');
		}

		if (!empty($createdirifnotexists)) {
			if (dol_mkdir($upload_dir) < 0) { // needed by products
				throw new RestException(500, 'Error while trying to create directory '.$upload_dir);
			}
		}


		$destfile = $upload_dir.'/'.$original_file;
		$destfiletmp = DOL_DATA_ROOT.'/admin/temp/'.$original_file;

		$res = dol_delete_file($destfiletmp);
		

		if (!dol_is_dir(dirname($destfile))) {
			throw new RestException(400, 'Directory does not exists : '.dirname($destfile));
		}

		if (!$overwriteifexists && dol_is_file($destfile)) {
			throw new RestException(400, "File with name '".$original_file."' already exists.");
		}

		// in case temporary directory admin/temp doesn't exist
		if (!dol_is_dir(dirname($destfiletmp))) {
			dol_mkdir(dirname($destfiletmp));
		}

		$fhandle = @fopen($destfiletmp, 'w');
		if ($fhandle) {
			$nbofbyteswrote = fwrite($fhandle, $newfilecontent);
			fclose($fhandle);
			dolChmod($destfiletmp);
		} else {
			throw new RestException(500, "Failed to open file '".$destfiletmp."' for write");
		}

		$disablevirusscan = 0;
		$src_file = $destfiletmp;
		$dest_file = $destfile;

		// Security:
		// If we need to make a virus scan
		if (empty($disablevirusscan) && file_exists($src_file)) {
			$checkvirusarray = dolCheckVirus($src_file, $dest_file);
			if (count($checkvirusarray)) {
				dol_syslog('Files.lib::dol_move_uploaded_file File "'.$src_file.'" (target name "'.$dest_file.'") KO with antivirus: errors='.implode(',', $checkvirusarray), LOG_WARNING);
				throw new RestException(500, 'ErrorFileIsInfectedWithAVirus: '.implode(',', $checkvirusarray));
			}
		}

		// Security:
		// Disallow file with some extensions. We rename them.
		// Because if we put the documents directory into a directory inside web root (very bad), this allows to execute on demand arbitrary code.
		if (isAFileWithExecutableContent($dest_file) && !getDolGlobalString('MAIN_DOCUMENT_IS_OUTSIDE_WEBROOT_SO_NOEXE_NOT_REQUIRED')) {
			// $upload_dir ends with a slash, so be must be sure the medias dir to compare to ends with slash too.
			$publicmediasdirwithslash = $conf->medias->multidir_output[$conf->entity];
			if (!preg_match('/\/$/', $publicmediasdirwithslash)) {
				$publicmediasdirwithslash .= '/';
			}

			if (strpos($upload_dir, $publicmediasdirwithslash) !== 0 || !getDolGlobalInt("MAIN_DOCUMENT_DISABLE_NOEXE_IN_MEDIAS_DIR")) {	// We never add .noexe on files into media directory
				$dest_file .= '.noexe';
			}
		}

		// Security:
		// We refuse cache files/dirs, upload using .. and pipes into filenames.
		if (preg_match('/^\./', basename($src_file)) || preg_match('/\.\./', $src_file) || preg_match('/[<>|]/', $src_file)) {
			dol_syslog("Refused to deliver file ".$src_file, LOG_WARNING);
			throw new RestException(500, "Refused to deliver file ".$src_file);
		}

		// Security:
		// We refuse cache files/dirs, upload using .. and pipes into filenames.
		if (preg_match('/^\./', basename($dest_file)) || preg_match('/\.\./', $dest_file) || preg_match('/[<>|]/', $dest_file)) {
			dol_syslog("Refused to deliver file ".$dest_file, LOG_WARNING);
			throw new RestException(500, "Refused to deliver file ".$dest_file);
		}

		$moreinfo = array('note_private' => 'File uploaded using API /documents from IP '.getUserRemoteIP());
		if (!empty($object) && is_object($object) && $object->id > 0) {
			$moreinfo['src_object_type'] = $object->table_element;
			$moreinfo['src_object_id'] = $object->id;
		}

		// Move the temporary file at its final emplacement
		$result = dol_move($destfiletmp, $dest_file, 0, $overwriteifexists, 1, 1, $moreinfo);
		if (!$result) {
			throw new RestException(500, "Failed to move file into '".$dest_file."'");
		}

		//WE save file upladed to DB if loaded in lib folder
		if($subdir == "lib" && $sbiuserid){
			$fileLibObject = new FilesLib($this->db);
			$fileLibObject->label = dol_basename($destfile);
			$fileLibObject->date_creation = dol_now();
			$fileLibObject->fk_user_creat = $sbiuserid;
			$fileLibObject->create(DolibarrApiAccess::$user);
		}

		return dol_basename($destfile);
	}


	/**
	 * Transfer documents to Dolibarr element from transaction folder
	 *
	 * @param int   $id             Id of transaction
	 *
	 * @url	POST transactions/{id}/document/transfer
	 *
	 * @return int|bool
	 *
	 * @throws RestException 403
	 * @throws RestException 404
	 */
	public function transfer_documents($id)
	{
		global $conf;

		if(empty(DolibarrApiAccess::$user->hasRight("smartbankimport", "write"))){
			throw new RestException(403, 'User not allowed');
			return -1;
		}

		if(empty($id)){
			throw new RestException(400, 'bad value for parameter id');
			return -1;
		}

		$transactionObject = new TransactionsObj($this->db);

		$resTransaction = $transactionObject->fetch($id);

		if($resTransaction <= 0){
			throw new RestException(400, 'Cant find transaction');
			return -1;
		}

		$elementTypeAllowed = array(
			"payment",
			"payment_supplier", 
			"paiementcharge", 
			"payment_loan",
			"payment_vat", 
			"payment_salary", 
			"payment_expensereport", 
			"payment_various"
		);

		if(!in_array($transactionObject->import_element_type, $elementTypeAllowed)){
			throw new RestException(500, "Unknow element type : ".$transactionObject->import_element_type);
			return -1;
		}

		//GET PAIEMENT
		//-----------------------------------------------------------------------------------------------
		//-----------------------------------------------------------------------------------------------

		$requestResult = $this->getPayments($transactionObject->import_element_type, "rowid", 'DESC', 100, 0, '(p.rowid:=:'.$transactionObject->import_element_id.')');

		if(!is_array($requestResult)){
			return $requestResult;
		}

		if(empty($requestResult[0])){
			return $requestResult;
		}


		//PREPARE REF OF ELEMENT
		//-----------------------------------------------------------------------------------------------
		//-----------------------------------------------------------------------------------------------
		$fullEndPoint = "";

		if($transactionObject->import_element_type == "payment"){
			require_once DOL_DOCUMENT_ROOT.'/compta/facture/class/facture.class.php';
			$object = new Facture($this->db);
			$result = $object->fetch($requestResult[0]->fk_facture);
			if (!$result) {
				throw new RestException(404, 'Invoice not found');
			}

			$upload_dir = $conf->facture->dir_output."/".get_exdir(0, 0, 0, 1, $object, 'invoice');

		}elseif($transactionObject->import_element_type == "payment_supplier"){

			require_once DOL_DOCUMENT_ROOT.'/fourn/class/fournisseur.facture.class.php';

			$object = new FactureFournisseur($this->db);
			$result = $object->fetch($requestResult[0]->fk_facturefourn);
			if (!$result) {
				throw new RestException(404, 'Invoice not found');
			}

			//$tmpreldir = get_exdir($object->id, 2, 0, 0, $object, 'invoice_supplier');

			$upload_dir = $conf->fournisseur->dir_output."/facture/".get_exdir($object->id, 2, 0, 0, $object, 'invoice_supplier').dol_sanitizeFileName($object->ref);

		}elseif($transactionObject->import_element_type == "paiementcharge"){

			require_once DOL_DOCUMENT_ROOT.'/compta/sociales/class/chargesociales.class.php';

			$object = new ChargeSociales($this->db);

			$result = $object->fetch($requestResult[0]->fk_charge);
			if (!$result) {
				throw new RestException(404, 'Charge not found');
			}

			$upload_dir = $conf->tax->dir_output.'/'.dol_sanitizeFileName($object->ref);

		}elseif($transactionObject->import_element_type == "payment_vat"){
			require_once DOL_DOCUMENT_ROOT.'/compta/tva/class/tva.class.php';
			$object = new Tva($this->db);

			$result = $object->fetch($requestResult[0]->fk_tva);
			if (!$result) {
				throw new RestException(404, 'VAT not found');
			}

			$upload_dir = $conf->tax->dir_output.'/vat/'.dol_sanitizeFileName($object->ref);

		}elseif($transactionObject->import_element_type == "payment_salary"){
			require_once DOL_DOCUMENT_ROOT.'/salaries/class/salary.class.php';

			$object = new Salary($this->db);

			$result = $object->fetch($requestResult[0]->fk_salary);
			if (!$result) {
				throw new RestException(404, 'Salary not found');
			}

			$upload_dir = $conf->salaries->dir_output.'/'.dol_sanitizeFileName($object->id);

		}elseif($transactionObject->import_element_type == "payment_expensereport"){

			require_once DOL_DOCUMENT_ROOT.'/expensereport/class/expensereport.class.php';

			$object = new ExpenseReport($this->db);
			$result = $object->fetch($requestResult[0]->fk_expensereport);
			if (!$result) {
				throw new RestException(404, 'Expense report not found');
			}

			$upload_dir = $conf->expensereport->dir_output.'/'.dol_sanitizeFileName($object->ref);

		}elseif($transactionObject->import_element_type == "payment_various"){

			require_once DOL_DOCUMENT_ROOT.'/compta/bank/class/paymentvarious.class.php';

			$object = new PaymentVarious($this->db);
			$result = $object->fetch($requestResult[0]->rowid);
			if (!$result) {
				throw new RestException(404, 'Payment various not found');
			}

			$upload_dir = $conf->bank->dir_output.'/'.dol_sanitizeFileName($object->id);
		
		}else{
			throw new RestException(500, "Unknow element type : ".$transactionObject->import_element_type);
			return -1;
		}

		if(empty($transactionObject->fk_gocardless_bankaccount)){
			throw new RestException(500, "Could not find account folder number for transaction : ".$id);
		}


		$upload_dir_origin = dol_sanitizePathName($conf->smartbankimport->dir_output . "/" . $transactionObject->fk_gocardless_bankaccount . "/" . $id);
		$utf8_path = $upload_dir_origin;
		$types = "files";
		$recursive = 1;
		$filter = "";
		$excludefilter = ".json";
		$sortcriteria = "name";

		$dirList = dol_dir_list($utf8_path, $types, $recursive, $filter, $excludefilter, $sortcriteria);

		$response = array();

		if(!empty($dirList)){
			foreach ($dirList as $file) {

				//GET BANKIMPORT DOCUMENT
				$filename = dol_sanitizeFileName($file["name"]);
				$original_file_osencoded = dol_osencode($upload_dir_origin . "/" . $filename);

				if (!file_exists($original_file_osencoded)) {
					throw new RestException(500, "File not found : " . $original_file_osencoded);
					return -1;
				}

				$dest_file = $upload_dir . "/" . $filename;

				// Move the temporary file at its final emplacement
				$overwriteifexists = 1;

				$moreinfo = array('note_private' => 'File uploaded using Smart Bank Import ');
				if (!empty($object) && is_object($object) && $object->id > 0) {
					$moreinfo['src_object_type'] = $object->table_element;
					$moreinfo['src_object_id'] = $object->id;
				}

				//WE CREATE THE FOLDER IF NECESSARY
				if (!dol_is_dir(dirname($dest_file))) {
					dol_mkdir(dirname($dest_file));
				}

				$result = dol_move($original_file_osencoded, $dest_file, 0, $overwriteifexists, 1, 1, $moreinfo);
				
				if (!$result) {
					throw new RestException(500, "Failed to move file into '".$dest_file."'");
				}


				$response[] = array(
					"filename" => $filename,
					"modulepart" => $object->table_element,
					"id" => $object->id
				);
			}
		}

		return $response;
	}

	/**
	 * Delete a document.
	 *
	 * @param   string  $original_file  Relative path with filename, relative to modulepart (for example: PRODUCT-REF-999/IMAGE-999.jpg)
	 * @return  array                   List of documents
	 *
	 * @url DELETE /deletefile
	 *
	 * @throws	RestException	400  Bad value for parameter modulepart
	 * @throws	RestException	400  Bad value for parameter original_file
	 * @throws	RestException	403  Access denied
	 * @throws	RestException	404  File not found
	 * @throws	RestException	500  Error on file operation
	 */
	public function deletefile($original_file)
	{
		global $conf, $langs;


		if (empty($original_file)) {
			throw new RestException(400, 'bad value for parameter original_file');
		}


		if (!DolibarrApiAccess::$user->hasRight('smartbankimport', 'write')) {
			throw new RestException(403);
		}


		$relativePathArray = explode("/",$original_file);

		$entity = !empty(DolibarrApiAccess::$user->entity) ? DolibarrApiAccess::$user->entity : 1;

		$original_file = $conf->smartbankimport->multidir_output[$entity].'/'.$original_file;


		if (preg_match('/\.\./', $original_file) || preg_match('/[<>|]/', $original_file)) {
			throw new RestException(403);
		}

		$filename = basename($original_file);
		$original_file_osencoded = dol_osencode($original_file); // New file name encoded in OS encoding charset

		if (!file_exists($original_file_osencoded)) {
			dol_syslog("Try to download not found file ".$original_file_osencoded, LOG_WARNING);
			throw new RestException(404, 'File not found');
		}

		if (@unlink($original_file_osencoded)) {
			return array(
				'success' => array(
					'code' => 200,
					'message' => 'Document deleted'
				)
			);
		}

		throw new RestException(403);
	}


}
